#version 330 core
#extension GL_ARB_explicit_attrib_location: enable

in vec3 vertexPosition;
flat in vec4 rgbaFog;

uniform float fogDensityIn;
uniform float fogMinIn;
uniform float dayLight;
uniform float horizonFog;
uniform vec3 playerPos;
uniform vec3 sunPosition;

uniform float cloudCounter; // Cloud update counter
uniform float cloudPixelation; // How blocky the clouds look
uniform float cloudPrecipitation; // 0 = clear, 1 = heavy precipitation
const float clouddarkClear = 0.2; // Default cloud darkness for clear weather
const float clouddarkStorm = 0.01; // Darker cloud darkness for stormy weather

// New Uniforms for cloud parameters
uniform float cloudScale; // The scale of the clouds
uniform float cloudAlpha; // Transparency of the clouds
uniform float cloudCover; // Cloud density

// adapted from https://www.shadertoy.com/view/4tdSWr
uniform float cloudSpeed = 0.01;
const float cloudlight = 0.2;
const float skytint = 0.6;

layout(location = 0) out vec4 outColor;
layout(location = 1) out vec4 outGlow;
#if SSAOLEVEL > 0
layout(location = 2) out vec4 outGNormal;
layout(location = 3) out vec4 outGPosition;
#endif

#include dither.fsh
#include fogandlight.fsh
#include skycolor.fsh
#include noise3d.ash

// https://github.com/hughsk/glsl-luma/blob/master/index.glsl
float luma(vec3 color) {
  return dot(color, vec3(0.299, 0.587, 0.114));
}

const mat2 m = mat2( 1.6,  1.2, -1.2,  1.6 );

vec2 hash_( vec2 p ) {
	p = vec2(dot(p,vec2(127.1,311.7)), dot(p,vec2(269.5,183.3)));
	return -1.0 + 2.0*fract(sin(p)*43758.5453123);
}

float noise_( in vec2 p ) {
    const float K1 = 0.366025404; // (sqrt(3)-1)/2;
    const float K2 = 0.211324865; // (3-sqrt(3))/6;
	vec2 i = floor(p + (p.x+p.y)*K1);	
    vec2 a = p - i + (i.x+i.y)*K2;
    vec2 o = (a.x>a.y) ? vec2(1.0,0.0) : vec2(0.0,1.0); //vec2 of = 0.5 + 0.5*vec2(sign(a.x-a.y), sign(a.y-a.x));
    vec2 b = a - o + K2;
	vec2 c = a - 1.0 + 2.0*K2;
    vec3 h = max(0.5-vec3(dot(a,a), dot(b,b), dot(c,c) ), 0.0 );
	vec3 n = h*h*h*h*vec3( dot(a,hash_(i+0.0)), dot(b,hash_(i+o)), dot(c,hash_(i+1.0)));
    return dot(n, vec3(70.0));	
}

float fbm(vec2 n) {
	float total = 0.0, amplitude = 0.1;
	for (int i = 0; i < 7; i++) {
		total += noise_(n) * amplitude;
		n = m * n;
		amplitude *= 0.4;
	}
	return total;
}

// Get cloud color based on dynamic uniforms
vec4 getCloudColor(vec4 skyColor) {

    vec2 iResolution = vec2(128); // screen resolution
    float iTime = cloudCounter;

    // Get the original fragment position and scale it to the desired pixelation level
    vec2 p = vertexPosition.xz / iResolution.xy;
    p = floor(p * cloudPixelation) / cloudPixelation; // Higher value = less pixelated, might want to flip this logic...

    vec2 uv = p * vec2(iResolution.x / iResolution.y, 1.0);    
    float time = iTime * cloudSpeed;
    float q = fbm(uv * cloudScale);
    
    // Ridged noise shape
    float r = 0.0;
    uv *= cloudScale;
    uv -= q - time;
    float weight = 0.8;
    for (int i = 0; i < 8; i++) {
        r += abs(weight * noise_(uv));
        uv = m * uv + time;
        weight *= 0.7;
    }
    
    // Noise shape
    float f = 0.0;
    uv = p * vec2(iResolution.x / iResolution.y, 1.0);
    uv *= cloudScale;
    uv -= q - time;
    weight = 0.7;
    for (int i = 0; i < 8; i++) {
        f += weight * noise_(uv);
        uv = m * uv + time;
        weight *= 0.6;
    }
    
    f *= r + f;
    
    // Noise color
    float c = 0.0;
    time = iTime * cloudSpeed * 2.0;
    uv = p * vec2(iResolution.x / iResolution.y, 1.0);
    uv *= cloudScale * 2.0;
    uv -= q - time;
    weight = 0.4;
    for (int i = 0; i < 7; i++) {
        c += weight * noise_(uv);
        uv = m * uv + time;
        weight *= 0.6;
    }
    
    // Noise ridge color
    float c1 = 0.0;
    time = iTime * cloudSpeed * 3.0;
    uv = p * vec2(iResolution.x / iResolution.y, 1.0);
    uv *= cloudScale * 3.0;
    uv -= q - time;
    weight = 0.4;
    for (int i = 0; i < 7; i++) {
        c1 += abs(weight * noise_(uv));
        uv = m * uv + time;
        weight *= 0.6;
    }
    
    c += c1;

    float clouddark = mix(clouddarkClear, clouddarkStorm, cloudPrecipitation);

    vec3 cloudcolour = vec3(1.1, 1.1, 0.9) * clamp((clouddark + cloudlight * c), 0.0, 1.0);
    
    f = cloudCover + cloudAlpha * f * r;
    
    vec3 result = mix(skyColor.rgb, clamp(skytint * skyColor.rgb + cloudcolour, 0.0, 1.0), clamp(f + c, 0.0, 1.0));
    
    return vec4(result, 1.0);
}

void main()
{
    outColor = vec4(1);
    outGlow = vec4(1);
    float sealevelOffsetFactor = 0.25;

    // Get sky color and glow
    getSkyColorAt(vertexPosition, sunPosition, sealevelOffsetFactor, clamp(dayLight, 0, 1), horizonFog, outColor, outGlow);

    // Get the cloud color
    vec4 cloudColor = getCloudColor(outColor);

    // Blend cloud color with the sky color based on dayLight
    // Only a small amount of the sky color is mixed into the cloud color as dayLight decreases
    float cloudBlendFactor = (1.0 - clamp(dayLight, 0, 1)) * 0.75;

    cloudColor = mix(cloudColor, outColor, cloudBlendFactor);

    // Mix the cloud color with the final outColor
    outColor = mix(outColor, cloudColor, clamp((vertexPosition.y) * 0.02, 0, 1));

    // Adjust glow
    outGlow.y *= clamp((dayLight - 0.05) * luma(outColor.rgb) * 0.2, 0, 1);

    #if SSAOLEVEL > 0
    outGPosition = vec4(0);
    outGNormal = vec4(0);
    #endif
}
